/******************************************************************************
 *
 * adc_ads1278.c - Drivers for the TI ads1278 ADC.
 *  Interfaces the TI tiva C Launchpad with the TI ads1278 analog to digital
 *  converter.
 *
 *  Author: Curtis Mayberry
 *  Georgia Tech IMEMS
 *  rev1 Feb 2014
 *
 *  Originally written for the MRIG gyroscope project
 *
 *  GT BE Peripherals:
 *   ADC: ADS1278
 *    INPUTS
 *     TEST0		-> GPIO PE4: PE4
 *     TEST1		-> GPIO PE4: PE4
 *     CLKDIV		-> GPIO PE5: PE5
 *     ~SYNC		-> GPIO PE0: PE0
 *     CLK			-> M1PWM5  : PF1 (option: hardware jumper)
 *     CLK			-> SSI2_CLK: PB4 (option: hardware jumper)
 *     MODE0		-> GPIO PA6: PA6
 *	   MODE1		-> GPIO PA7: PA7
 *	   FORMAT0		-> GPIO PC4: PC4
 *	   FORMAT1		-> GPIO PC5: PC5
 *	   FORMAT2		-> GPIO PC6: PC6
 *	   SERIAL
 *	   SCLK  		-> SSI2_CLK: PB4
 *	   ~DRDY/ FSYNC -> GPIO PB5: PB5 (CS) (SPI Format: ~DRDY)
 *	   DOUT1        -> SSI2_RX:  PB6 (MISO)
 *	   DOUT2		-> GPIO PB0: PB2
 *	   DOUT3		-> GPIO PB1: PB1
 *	   DOUT4		-> GPIO PB2: PB2
 *	   DOUT5		-> GPIO PB3: PB3
 *	   DOUT6		-> GPIO PB7: PB7
 *	   DOUT7		-> GPIO PD6: PD6
 *	   DOUT8		-> GPIO PD7: PD7
 *    OUTPUTS
 *	   NONE
 *	  CLK Generation
 *	   Timer 3
 *	  Synchronization
 *	   Timer 4
 *	 DAC
 *	  Timer 2 for synchronization
 *	 RGB Control
 * 		- Wide Timer 5B for blinking the entire RGB unit.
 * 		- Timer 0B intensity of an RGB element
 * 		- Timer 1A intensity of an RGB element
 * 		- Timer 1B intensity of an RGB element
 *
 *  This work is licensed under the Creative Commons Attribution-ShareAlike 3.0
 *  Unported License. To view a copy of this license, visit
 *  http://creativecommons.org/licenses/by-sa/3.0/ or send a letter to Creative
 *  Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 *
 ******************************************************************************/

#include <stdbool.h>
#include <stdint.h>
#include "driverlib/rom.h"
#include "inc/hw_memmap.h"
#include "inc/hw_ssi.h"
#include "inc/hw_types.h"
#include "inc/hw_ints.h"

// Tivaware
#include "driverlib/rom.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "driverlib/pin_map.h"
#include "driverlib/gpio.h"
#include "driverlib/pwm.h"
#include "driverlib/interrupt.h"
#include "driverlib/udma.h"

// GTBE Lib
#include "adc_ads1278.h"

/*****************
 * ADC Functions *
 *****************/

/**
 * Initializes the ADC
 *
 *  Gives clk freq = 25 MHz assuming a system clk freq = 50MHz
 *
 *  /param modeValue Sets the mode of the ADC
 *  values: ADC_MODE_HIGH_SPEED, ADC_MODE_HIGH_RESOL, ADC_MODE_LOW_POWER,
 *   ADC_MODE_LOW_SPEED
 *
 **/
 void ADC_initADC(void) {
	 ADC_initControls();
	 ADC_setTestMode(ADC_TEST_MODE_NORMAL);
	 ADC_initNSYNC();
	 ADC_setSerialFormat(ADC_FORMAT_SPI_TDM_DYN);
	 ADC_setCLKfreq(ADC_FREQ_37MHZ_HS);  // 37 MHz is max frequency
	 ADC_initClkM1PWM5(ADC_PWM_DIV2); // Sets actual device frequency
	 ADC_initSSI2();
 }

/**
 * Initializes the control signals of the ADC
 **/
 void ADC_initControls(void) {
	 // TEST0, TEST1, ~SYNC
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
	 ROM_GPIOPinTypeGPIOOutput(GPIO_PORTE_BASE, GPIO_PIN_5|GPIO_PIN_4|GPIO_PIN_0);
	 // MODE0, MODE1
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
	 ROM_GPIOPinTypeGPIOOutput(GPIO_PORTA_BASE, GPIO_PIN_7|GPIO_PIN_6);
	 // FORMAT2, FORMAT1, FORMAT0
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);
	 ROM_GPIOPinTypeGPIOOutput(GPIO_PORTC_BASE, GPIO_PIN_6|GPIO_PIN_5|GPIO_PIN_4);
 }

/**
 * Initialize ~DRDY pin as an input interrupt to trigger the read.
 * \note need to add the interrupt prototype to the NVIC
 **/
 void ADC_initDRDYint(void) {
	// ~DRDY
  	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
  	while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));
  	ROM_GPIOPinTypeGPIOInput(GPIO_PORTB_BASE, GPIO_PIN_5);
  	while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));
  	ROM_GPIOIntTypeSet(GPIO_PORTB_BASE, GPIO_PIN_5, GPIO_FALLING_EDGE);
  	while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));
  	GPIOIntEnable(GPIO_PORTB_BASE, GPIO_INT_PIN_5);
  	while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));
  	IntEnable(INT_GPIOB);
  	while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOB));
 }

 /**
  * Initialize ~DRDY pin as an input
  **/
  void ADC_initDRDY(void) {
	// ~DRDY
   	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
   	ROM_GPIOPinTypeGPIOInput(GPIO_PORTB_BASE, GPIO_PIN_5);
  }
/**
 * Initializes ~SYNC
 **/
  void ADC_initNSYNC(void) {
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
	 ROM_GPIOPinTypeGPIOOutput(GPIO_PORTE_BASE, GPIO_PIN_0);
	 ROM_GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_0, 0x01);
  }
/**
 * initializes SSI2 for use with the DAC
 **/
 void ADC_initSSI2(void) {
 	uint32_t trashBin[1] = {0};

 	// Enable Peripherals
 	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI2);
 	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);

 	// Set the pin muxing to SSI2 on port B
 	ROM_GPIOPinConfigure(GPIO_PB4_SSI2CLK);

 	ROM_GPIOPinConfigure(GPIO_PB6_SSI2RX);
 	ROM_GPIOPinConfigure(GPIO_PB7_SSI2TX);
 	//ROM_GPIOPinConfigure(GPIO_PB5_SSI2FSS);

 	ROM_GPIOPinTypeSSI(GPIO_PORTB_BASE,GPIO_PIN_7|GPIO_PIN_6|GPIO_PIN_4);

 	// SPI Mode1
 	ROM_SSIConfigSetExpClk(SSI2_BASE,SysCtlClockGet(),SSI_FRF_MOTO_MODE_1,
 	 						   SSI_MODE_MASTER, FREQ_SSI2_DATA, 8);
 	// Enable SSI uDMA
 	ROM_SSIEnable(SSI2_BASE);
 	//SSIDMAEnable(SSI2_BASE, SSI_DMA_RX);
 	while (ROM_SSIDataGetNonBlocking(SSI2_BASE, &trashBin[0])) {
 	     }
 	//ROM_IntEnable(INT_SSI0);
 }

/**
 * Sets the test mode of the ADC that tests the digital I/O
 *
 *  /param modeValue Sets the test mode of the ADC
 *  values: ADC_TEST_MODE_NORMAL, ADC_TEST_MODE_TEST
 **/
 void ADC_setTestMode(uint32_t testModeValue) {
	 ROM_GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_4, testModeValue);
 }

 /**
  * Sets the clock divider setting of the ADC
  *
  *  /param clkDivValue Sets the clock divider setting
  *  values: ADC_CLKDIV_HIGHF, ADC_CLKDIV_LOWF
  **/
 void ADC_setCLKfreq(uint32_t clkFreqValue) {
 	 switch(clkFreqValue) {
 	 case ADC_FREQ_37MHZ_HS :
 		ADC_setMode(ADC_MODE_HIGH_SPEED);
 		ADC_setCLKdiv(ADC_CLKDIV_HIGHF);
 		break;
 	 case ADC_FREQ_27MHZ_HR :
 		ADC_setMode(ADC_MODE_HIGH_RESOL);
 		ADC_setCLKdiv(ADC_CLKDIV_HIGHF);
 		break;
 	 case ADC_FREQ_27MHZ_LP :
 		ADC_setMode(ADC_MODE_LOW_POWER);
 		ADC_setCLKdiv(ADC_CLKDIV_HIGHF);
 		break;
 	 case ADC_FREQ_13p5MHZ_LP :
 		ADC_setMode(ADC_MODE_LOW_POWER);
 		ADC_setCLKdiv(ADC_CLKDIV_LOWF);
 		break;
 	 case ADC_FREQ_27MHZ_LS :
 		ADC_setMode(ADC_MODE_LOW_SPEED);
 		ADC_setCLKdiv(ADC_CLKDIV_HIGHF);
 		break;
 	 case ADC_FREQ_5p4MHZ_LS :
 		ADC_setMode(ADC_MODE_LOW_SPEED);
 		ADC_setCLKdiv(ADC_CLKDIV_LOWF);
 		break;
 	 default :
  		ADC_setMode(ADC_MODE_HIGH_SPEED);
  		ADC_setCLKdiv(ADC_CLKDIV_HIGHF);
 	 }
 }

/**
 * Sets the mode of the ADC
 *
 * 		Mode		   | MAX Fdata |
 * ---------------------------------
 * ADC_MODE_HIGH_SPEED |  144,531  |
 * ADC_MODE_HIGH_RESOL |   52,734  |
 * ADC_MODE_LOW_POWER  |   52,734  |
 * ADC_MODE_LOW_SPEED  |   10,547  |
 *
 *  /param modeValue Sets the mode of the ADC
 *  values: ADC_MODE_HIGH_SPEED, ADC_MODE_HIGH_RESOL, ADC_MODE_LOW_POWER,
 *   ADC_MODE_LOW_SPEED
 **/
 void ADC_setMode(uint32_t modeValue) {
	 ROM_GPIOPinWrite(GPIO_PORTA_BASE, GPIO_PIN_7|GPIO_PIN_6, modeValue);
 }

/**
 * Sets the clock divider setting of the ADC
 *
 *  /param clkDivValue Sets the clock divider setting
 *  values: ADC_CLKDIV_HIGHF, ADC_CLKDIV_LOWF
 **/
 void ADC_setCLKdiv(uint32_t CLKdivValue) {
	 ROM_GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_5, CLKdivValue);
 }

/**
 * Initializes the M1PWM5 to generate the clock for the ADC
 *  Utilizes PWM module 1 output 5 (PF1)
 *
 *  Clock Setup Options
 *  *25MHz    - 50 MHz system clock,    sysClkFreqDiv = ADC_PWM_DIV2
 *  *26.67MHz - 80 MHz system clock,    sysClkFreqDiv = ADC_PWM_DIV3
 *  *33.33MHz - 66.67 MHz system clock, sysClkFreqDiv = ADC_PWM_DIV2
 *	 (Need 2.0v < DVDD < 2.2v)
 *  /param sysClkFreqDiv - frequency divider of the PWM
 *  values: ADC_PWM_DIV2, ADC_PWM_DIV3
 **/
 void ADC_initClkM1PWM5(uint32_t sysClkFreqDiv) {
	// Setup GPIO config
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_PWM1);
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);

	ROM_GPIOPinTypePWM(GPIO_PORTF_BASE, GPIO_PIN_1);
	ROM_GPIOPinConfigure(GPIO_PF1_M1PWM5);
	ROM_SysCtlPWMClockSet(SYSCTL_PWMDIV_1);

	// Configure the PWM generator for count down mode with immediate updates
	// to the parameters.
	ROM_PWMGenConfigure(PWM1_BASE, PWM_GEN_2,
			            PWM_GEN_MODE_DOWN | PWM_GEN_MODE_NO_SYNC | PWM_GEN_MODE_DBG_STOP);

	// Set the period in number of clock cycles.
	//PWMGenPeriodSet(PWM1_BASE, PWM_GEN_2, sysClkFreqDiv); // Does not work
	//PWMGenPeriodSet(PWM1_BASE, PWM_GEN_2, 2); // Does not work
	ROM_PWMGenPeriodSet(PWM1_BASE, PWM_GEN_2, 4);  //works freq = 20MHz
	//PWMGenPeriodSet(PWM1_BASE, PWM_GEN_2, 8);  //works freq = 10MHz

	// Set the pulse width.
	//PWMPulseWidthSet(PWM1_BASE, PWM_OUT_5, 1); // Does not work
	ROM_PWMPulseWidthSet(PWM1_BASE, PWM_OUT_5, 2); //works freq = 20MHz
	//PWMPulseWidthSet(PWM1_BASE, PWM_OUT_5, 4); //works freq = 10MHz

	// Start the timer.
	ROM_PWMGenEnable(PWM1_BASE, PWM_GEN_2);
	// Enable the output.
	ROM_PWMOutputState(PWM1_BASE,  PWM_OUT_5_BIT, true);
 }

void ADC_setSerialFormat(uint32_t serialFormatValue) {
	ROM_GPIOPinWrite(GPIO_PORTC_BASE, GPIO_PIN_6 | GPIO_PIN_5 | GPIO_PIN_4, serialFormatValue);
}
/**
 * Initializes the SSI2 RX uDMA to transfer 3 bytes from the fifo to the data buffer
 **/
void ADC_initUDMAssi2RX(void) {
	SSIDMAEnable(SSI2_BASE, SSI_DMA_RX);

	uDMAChannelAssign(UDMA_CH12_SSI2RX);

	// Place the uDMA channel attributes in a known state. These should already be disabled by default.
	uDMAChannelAttributeDisable(UDMA_CHANNEL_SSI2RX,
	                            UDMA_ATTR_USEBURST | UDMA_ATTR_ALTSELECT |
	                            (UDMA_ATTR_HIGH_PRIORITY |
	                            UDMA_ATTR_REQMASK));
	// Configure the control parameters for the SSI2 RX channel.  The channel
	// will be used to transfer the ADC measurements to memory.
	uDMAChannelControlSet(UDMA_CHANNEL_SSI2RX | UDMA_PRI_SELECT,
						  UDMA_SIZE_8 | UDMA_SRC_INC_NONE | UDMA_DST_INC_8 |
						  UDMA_ARB_4);
	// Set up the transfer parameters for the SSI2 Rx channel.  This will
	// configure the transfer buffers and the transfer size.
	uDMAChannelTransferSet(UDMA_CHANNEL_SSI2RX | UDMA_PRI_SELECT,
						   UDMA_MODE_BASIC,
						   (void *)(SSI2_BASE + SSI_O_DR), ADC_g_dataBufferBytes,
						   3);
	uDMAChannelAttributeEnable(UDMA_CHANNEL_SSI2RX, UDMA_ATTR_HIGH_PRIORITY);
	SSIIntEnable(SSI2_BASE, SSI_DMARX);
	IntEnable(INT_SSI2);

	uDMAChannelEnable(UDMA_CHANNEL_SSI2RX);
}

void ADC_SSI2RXuDMA_ISR(void) {
	//uint32_t interruptStatus;
	//interruptStatus = ROM_SSIIntStatus(SSI2_BASE, 1);
	//ROM_SSIIntClear(SSI2_BASE, interruptStatus);
	uDMAChannelTransferSet(UDMA_CHANNEL_SSI2RX | UDMA_PRI_SELECT,
						   UDMA_MODE_BASIC,
						   (void *)(SSI2_BASE + SSI_O_DR), ADC_g_dataBufferBytes,
						   3);
	ROM_uDMAChannelEnable(UDMA_CHANNEL_SSI2RX);
	DAC_g_dataReady = true;
}
